
/*
 * editfoil.c - read in an airfoil and edit camber and thickness.
 *
 * Copyright 1994 Shamim P. Mohamed
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author:
 * shamim@math.isu.edu
 * Shamim Mohamed
 * Dept. of Mathematics
 * Idaho State University
 *
 */

#include "editfoil.h"
static char copyright[] = "\
Editfoil v1.0, Copyright 1994 Shamim Mohamed. This is free software\n\
and is freely distributable under certain conditions; it coms with\n\
NO WARRANTY. See the file \"COPYING\" for details.\n";

static char usage_str[] = "\n\
Usage: %s [-hq] [-c <camber>] [-t <thick>] [-o output-file] [datafile]\n\
\n\
Options:\n\
   -q : Don't print the copyright message\n\
   -h : Print this help\n\
These take arguments:\n\
   -c : Make the camber this value\n\
   -t : Make the thickness this value\n\
   -o : Save the data in this file instead of sending to stdout\n\
(Under MS-DOS, / can be used for options instead of -.)\n";

int main(int argc, char *argv[])
{
   int qflag=0, tflag=0, cflag=0;

   float t=-1., c=-1.;
   FILE *fp=stdin, *fout=stdout;
   int argp = 1;
   airfoil_data_t foil, newfoil;
   char *fname = "<stdin>", *p;
   
   while(argc > argp) {
      p = argv[argp];
      if(optionchar(*p) && (*(p+1) != 0)) {
         switch (*(p+1)) {
	 case 't':
	    if(sscanf(p = argv[++argp], "%f", &t) != 1 || t < 0.) {
	       fprintf(stderr, "Invalid argument for -t: \"%s\"\n", p);
	       break;
	    }
	    tflag++;
	    t /= 100;
	    break;
	 case 'c':
	    if(sscanf(p = argv[++argp], "%f", &c) != 1 || c < 0.) {
	       fprintf(stderr, "Invalid argument for -c: \"%s\"\n", p);
	       break;
	    }
	    cflag++;
	    c /= 100;
	    break;
         case 'o':
            fname = argv[++argp];
            if((fout = fopen(fname, "w")) == NULL) {
               fprintf(stderr, "Could not open \"%s\" for writing!\n", fname);
               fout = stdout;
            }
            break;
         case '-':
            goto done_opts;
         case 'h':
         case '?':
         default:
            fprintf(stderr, usage_str, argv[0], argv[0]);
            exit(1);
         }
      }
      else
         break;
      argp++;
   }

 done_opts:
   
   if(!qflag)
      fputs(copyright, stderr);

   if(argp < argc)
      if((fp = fopen((fname = argv[argp]), "r")) == NULL) {
	 fprintf(stderr, "%s: can't open \"%s\"\n", argv[0], fname);
	 exit(1);
      }
   
   if(!read_foil(fp, fname, &foil))
      exit(1);
   
   edit(&newfoil, &foil, t, c);

   writefoil(fout, &newfoil, tflag, cflag);

   if(fp != stdin)
      fclose(fp);
   if(fout != stdout)
      fclose(fout);

   exit(0);

   /* well... */
   return 0;
}

static void edit(airfoil_data_t *new, airfoil_data_t *old, float t, float c)
{
   point_t other_side[MAXPOINTS];
   int i;

   get_otherside(other_side, old->points, old->npoints);

   /* find camber */
   old->camber = 0.0;
   for(i = 0; i < old->npoints; i++) {
      double mean;

      mean = 0.5 * (old->points[i].y + (double)(other_side[i].y));
      if(mean > old->camber) {
	 old->camber = mean;
	 old->cam_loc = old->points[i].x;
      }
   }

   /* find thickness */
   old->thickness = 0.0;
   for(i = 0; i < old->npoints; i++) {
      double thick;

      thick = (old->points[i].y - (other_side[i].y));
      thick = abs(thick);
      if(thick > old->thickness) {
	 old->thickness = thick;
	 old->th_loc = old->points[i].x;
      }
   }

   strcpy(new->title, old->title);
   new->npoints = old->npoints;
   for(i = 0; i < old->npoints; i++) {
      new->points[i].x = old->points[i].x;
      new->points[i].y = old->points[i].y;
   }
   new->miny = old->miny;
   new->maxy = old->maxy;
   
   /* De we need to modify camber? */
   if(c >= 0.) {
      double factor;
      
      if(old->camber < EPSILON) {
	 fputs("Cannot modify camber for a symmetrical airfoil!\n", stderr);
	 return;
      }

      factor = c / old->camber;
      for(i = 0; i < old->npoints; i++) {
	 double m = 0.5*(new->points[i].y + other_side[i].y);
	 double newm = m * factor;
	 new->points[i].y -= m - newm;
	 other_side[i].y -= m - newm;
      }
      new->camber = c;
   }
   else
      new->camber = old->camber;

   /* Modify thickness? */
   if (t >= 0.) {
      double factor = t / old->thickness;
      for(i = 0; i < new->npoints; i++) {
	 double m = 0.5*(new->points[i].y + other_side[i].y);
	 double th = (new->points[i].y - other_side[i].y);
	 double newt = th * factor;
	 new->points[i].y = m + newt/2.;
	 other_side[i].y = m - newt/2.;
      }
      new->thickness = t;
   }
   else
      new->thickness = old->thickness;
   
}
