
/*
 * mixfoils.c - read in two airfoils and generate an intermediate one
 *
 * (I don't know how useful this is... maybe there needs to be a way of
 * adding twist too, maybe I want a wing with 7032 at the root, 7037 at the
 * tips, uniform 3 deg. washout, and I want the section at the poly break
 * at 70% span. Maybe some other time.)
 *
 * Copyright 1994 Shamim P. Mohamed
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author:
 * shamim@math.isu.edu
 * Shamim Mohamed
 * Dept. of Mathematics
 * Idaho State University
 *
 */

#include "mixfoils.h"

static char copyright[] = "\
Mixfoil v1.0, Copyright 1994 Shamim Mohamed. This is free software\n\
and is freely distributable under certain conditions; it comes with\n\
NO WARRNATY. See the file \"COPYING\" for details.\n";

static char usage_str[] = "\n\
Usage: %s [-hq] [-o output-file] [-r ratio] section1 section2 \n\
\n\
Options:\n\
   -q : Don't print the copyright message\n\
   -h : Print this help\n\
These take arguments:\n\
   -r : ratio, between 0 and 100 (default is 50)\n\
(Under MS-DOS, / can be used for options instead of -.)\n";

static void merge(airfoil_data_t *new,
		  airfoil_data_t *foil1, airfoil_data_t *foil2, float ratio);

int main(int argc, char *argv[])
{
   int qflag=0;
   FILE *fp, *fout=stdout;
   int argp = 1;
   airfoil_data_t foil1, foil2, newfoil;
   char *fname, *p;
   float ratio = 50.;
   
   while(argc > argp) {
      p = argv[argp];
      if(optionchar(*p) && (*(p+1) != 0)) {
         switch (*(p+1)) {
	 case 'r':
            if(sscanf(p = argv[++argp], "%f", &ratio) != 1 ||
	       ratio < 0. || ratio > 100.) {
               fprintf(stderr, "Invalid argument for -r: \"%s\"\n", p);
               break;
            }
	 case 'q':
	    qflag++; break;
	 case 'o':
            fname = argv[++argp];
            if((fout = fopen(fname, "w")) == NULL) {
               fprintf(stderr, "Could not open \"%s\" for writing!\n", fname);
               fout = stdout;
            }
            break;

         case '-':
            goto done_opts;
         case 'h':
         case '?':
         default:
            fprintf(stderr, usage_str, argv[0], argv[0]);
            exit(1);
         }
      }
      else
         break;
      argp++;
   }

 done_opts:

   if(!qflag)
      fputs(copyright, stderr);

   if((argc - argp) != 2) {
      fprintf(stderr, "%s: %d need two filenames to merge!\n", argv[0], argp);
      exit(1);
   }

   if((fp = fopen((fname = argv[argp++]), "r")) == NULL) {
      fprintf(stderr, "%s: can't open \"%s\"\n", argv[0], fname);
      exit(1);
   }
   if(!read_foil(fp, fname, &foil1))
      exit(1);

   if((fp = fopen((fname = argv[argp++]), "r")) == NULL) {
      fprintf(stderr, "%s: can't open \"%s\"\n", argv[0], fname);
      exit(1);
   }
   if(!read_foil(fp, fname, &foil2))
      exit(1);

   ratio /= 100.0;
   merge(&newfoil, &foil1, &foil2, ratio);
   writefoil(fout, &newfoil, 0, 0);
   
   exit(0);

   /* well... */
   return 0;
}

static void merge(airfoil_data_t *new,
		  airfoil_data_t *foil1, airfoil_data_t *foil2, float ratio)
{
   airfoil_data_t *temp;
   point_t foil2normalised[MAXPOINTS];
   int i;

   if(foil1->npoints < foil2->npoints) {
      temp = foil1; foil1 = foil2; foil2 = temp;
      ratio = 1.0 - ratio;
   }
   
   new->npoints = foil1->npoints;
   sprintf(new->title, "%s (%.3g%%), %s (%.3g%%)", foil1->title, ratio*100,
	   foil2->title, (1.0-ratio)*100);

   for(i = 0; i < foil1->npoints; i++)
      foil2normalised[i].x = foil1->points[i].x;

   fill_in(foil2normalised, foil1->npoints, foil2->points, foil2->npoints);

   for(i = 0; i < foil1->npoints; i++) {
      new->points[i].x = foil1->points[i].x;
      new->points[i].y = ratio * foil2normalised[i].y +
	 (1.0 - ratio) * foil1->points[i].y;
   }
}

