/*
 * Copyright 1992,1994,1995 Shamim P. Mohamed
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author:
 * shamim@math.isu.edu
 * Shamim Mohamed
 * Dept. of Mathematics
 * Idaho State University
 *
 * Code reorganisation help from:
 * David Lindner
 * lindner@zk3.dec.com
 * DEC
 * ---------------------------------------------------------------------------
 *
 * Plots airfoils (or indeed any section)
 * ---------------------------------------------------------------------------
 * There are two formats for input data. Here are examples of each:
 *
 * Input formats:
 *    Title
 *    1.0	0
 *    0.9	0.011
 *    0.85	0.02
 *    ...
 *    1.0	0
 * This should start at the trailing edge (x = 1.0), follow a surface to the
 * LE (x = 0), and continue back to the TE (assuming the LE is smooth).
 * Or:
 *    Title
 *    0		0	0
 *    .0125	.0409   -.0183
 *    .025	.0529	-.0271
 *    ...
 *    1.0	0	0
 * The values go across the foil from LE (x = 0) to TE (x = 1).
 *
 * Lines beginning with # or % are comments, and are ignored.
 *
 * ---------------------------------------------------------------------------
 * Options:
 *    look for the string "usage_str".
 */

#include "plotfoil.h"
#include "externs.h"

/* global variables - these are modified by ui.c */

int sflag, iflag, tflag, nsheets_h, nsheets_v, nspars=0, aflag=0;
float s_limit = 0.025;
float c_height, c_depth;
float angle = 0.0;

value_t
   chord = {100.0, 2.834, 283.4, "mm"},
   sheeting = {0.0, 1.0, 0.0, ""},
   templ_h = {0.0, 1.0, 0.0, ""},
   templ_le = {0.0, 1.0, 0.0, ""},
   templ_te = {0.0, 1.0, 0.0, ""},
   xfudge = {0.0, 1.0, 0.0, ""},
   yfudge = {0.0, 1.0, 0.0, ""};

spar_t spars[MAX_SPARS];         /* spars for airfoil */
FILE *fout = stdout;             /* output fd */
int multipage = 0;               /* Is it a multipage plot? */

/*
 * function declarations
 */

static void plot_points(airfoil_data_t *);
static void get_value(char *, value_t *);
static void do_scaling(value_t *, value_t *);
static void process(FILE *, char *);

void plot(airfoil_data_t *);

static char usage_str[] = "\n\
Usage: %s [-hqs] [-c <n>[cim]] [-l <n>] [-i <n>] [-x <n>] [-y <n>]\n\
[-sp <offset> <thickness>] [-tt <n>] [-th <n>] [-tl <n>] [-o outfile] [files]\n\
   -h : Print this message\n\
   -q : Skip the copyright message at startup\n\
   -s : Use straight-line segments for x > s_limit\n\
These take arguments:\n\
   -x : Move the plot on the page to the right this much\n\
   -y : Move the plot up by this much\n\
   -c : Set chord size. A trailing unit can be specified:\n\
        c = cm, i = inches, m = mm (default)\n\
   -l : Set s_limit to n (default is 0.025)\n\
   -i : Draw sheeting allowance of n units (units same as chord)\n\
   -sp: Draw a spar at offset (in %% chord) of specified thickness\n\
   -a : Turn the section by this angle (deg.; positive is counter-clockwise)\n\
   -tl,\n\
   -tt,\n\
   -th: Draw template outlines of specified size (le, te, height)\n\
   -o : Write output to file\n\
(Under MS-DOS, / can be used instead of - for options.)\n\
\n\
A good place to start is:\n\
 %s -c 200m filename -o file.ps\n\n\
Typing just \"plotfoil\" will enter the interactive mode.\n\n";

int main(int argc, char *argv[])
{
   int i, qflag=0;
   char *p, *fname;
   FILE *fp, *get_file();
   int argp = 1;

   /* Special case: with no arguments, go into prompt mode. */
   if(argc == 1) {
      prompt_mode();
      exit(0);
   }
   
   while(argc > argp) {
      p = argv[argp];
      if(optionchar(*p) && (*(p+1) != 0)) {
	 switch (*(p+1)) {
	 case 'l':
	    sscanf(argv[++argp], "%f", &s_limit);
	    s_limit /= 100.0;
	    break;
	 case 's':
	    if(*(p+2) == 'p') {
	       sscanf(argv[++argp], "%f", &spars[nspars].offset);
	       get_value(argv[++argp], &spars[nspars].thick);
	       spars[nspars].offset /= 100.0;
	       nspars++;
	    }
	    else
	       sflag++;
	    break;
	 case 'q': qflag++; break;
	 case 'i':
	    iflag++;
	    get_value(argv[++argp], &sheeting);
	    break;
	 case 'c':
	    get_value(argv[++argp], &chord);
	    break;
	 case 'a':
	    aflag++;
	    sscanf(argv[++argp], "%f", &angle);
	    break;
	 case 't':
	    tflag++;
	    switch (*(p+2)) {
	    case 'h':
	       get_value(argv[++argp], &templ_h);
	       break;
	    case 't':
	       get_value(argv[++argp], &templ_te);
	       break;
	    case 'l':
	       get_value(argv[++argp], &templ_le);
	       break;
	    default:
	       fprintf(stderr, usage_str, argv[0], argv[0]);
	       break;
	    }
	    break;
	 case 'o':
	    fname = argv[++argp];
	    if((fout = fopen(fname, "w")) == NULL) {
	       fprintf(stderr, "Could not open \"%s\" for writing!\n", fname);
	       fout = stdout;
	    }
	    break;
	 case 'p':
	    prompt_mode();
	    exit(0);
	 case 'x':
	    get_value(argv[++argp], &xfudge);
	    break;
	 case 'y':
	    get_value(argv[++argp], &xfudge);
	    break;
	 case '-':
	    goto done_opts;
	 default:
	    fprintf(stderr, "Unknown option: %c%c\n", p[0], p[1]);
	 case 'h':
	 case '?':
	    fprintf(stderr, usage_str, argv[0], argv[0]);
	    exit(1);
	 }
      }
      else
	 break;
      argp++;
   }

 done_opts:
   
   if(!qflag)
      fputs(copyright, stderr);

   if(*chord.units == '\0') {
      chord.units = "mm";
      chord.scale = 2.834;
   }
   do_scaling(&chord, 0);
   do_scaling(&sheeting, &chord);
   do_scaling(&templ_h, &chord);
   do_scaling(&templ_te, &chord);
   do_scaling(&templ_le, &chord);
   do_scaling(&xfudge, &chord);
   do_scaling(&yfudge, &chord);
   for(i = 0; i < nspars; i++)
      do_scaling(&spars[i].thick, &chord);
   
   if(argp >= argc)
      process(stdin, "<stdin>");
   else
      for (; argp < argc; argp++) {
         if((fp = fopen(argv[argp], "r")) == NULL) {
            fprintf(stderr, "%s: can't open \"%s\"\n", argv[0], argv[argp]);
            continue;
         }
         process(fp, argv[argp]);
         fclose(fp);
      }

   exit(0);

   /* well... */
   return 0;
}

static void process(FILE *fp, char *fname)
{
   airfoil_data_t data;
   if(!(read_foil(fp, fname, &data)))
      return;
   plot(&data);
}
  
void get_value(char *s, value_t *val)
{
   char c;
   
   if(sscanf(s, "%f%c", &val->raw_value, &c) == 1) {
      val->scale = 1.0;
      val->units = "";
   }
   else
      switch((int)c) {
      case 'i': val->scale = 72.0; val->units = "\""; break;
      case 'c': val->scale = 28.34; val->units = "cm"; break;
      case 'm':
      default:
	 val->units = "mm";
	 val->scale = 2.834; break;
      }
   val->pts = val->raw_value * val->scale;
}

void do_scaling(val, master)
value_t *val, *master;
{
   if(val->units[0] == '\0' && master) {
      val->units = master->units;
      val->scale = master->scale;
   }
   val->pts = val->raw_value * val->scale;
}

void plot(airfoil_data_t *info)
{
   int vsize;
   
   c_height = info->maxy * chord.pts;
   c_depth = info->miny * chord.pts;
   if(tflag)
      vsize = 2*templ_h.pts;
   else
      vsize = (c_height - c_depth)*1.2;
   nsheets_v = vsize/PAGEHEIGHT + 1;
   nsheets_h = (int)(chord.pts+templ_te.pts+templ_le.pts)/PAGEWIDTH + 1;
   multipage = (nsheets_h > 1) || (nsheets_v > 1);

   init_output(info->title);
   plot_points(info);
   end_output(info->points, info->npoints);
}

/* Misc. math. */
double sqr(double x)
{
   return x*x;
}

#define px(i) (info->points[(i)].x)
#define py(i) (info->points[(i)].y)

static void plot_points(airfoil_data_t *info)
{
   int i, j;
   double d = 3.0, e = 0.2;

   if(tflag) {
      draw_template(templ_h.pts, templ_le.pts, templ_te.pts);
   }

   i = 0;

   if(sflag) {
      /* In the region s_limit to 100%, use straight lines to plot */
      i = 1;
      init_lines(px(0), py(0));
      while(px(i) > s_limit && px(i) < 1.0) {
         put_line(px(i), py(i));
         i++;
      }
      i--;
      
      for(j = i+1; px(j) <= s_limit && px(j) <= 1.0; j++)
	 ;
      
      interpolate(info->points+i, j-i-1, d, e, init_lines, put_curve);
      
      i = j;
      /* Finish off rest i.e. 2.5% to 100% with straight lines */
      for(;;) {
	 put_line(px(i), py(i));
	 if(px(i++) >= 1.0) break;
      }
   }
   else
      interpolate(info->points, info->npoints, d, e, init_lines, put_curve);

}

