
/*
 * This file is part of plotfoil v3
 * Copyright 1992,1994,1995 Shamim Mohamed
 *
 * ps_procs.c : PS procedures for plotfoil.
 *
 * warning: not for the faint of heart.
 */

#include "plotfoil.h"
#include "externs.h"

static void draw_spar(float offset, float thick, point_t *points, int n);
static void put_title(char *);

static char info[BUFSIZ];

static char *ps_header = "\
%%!PS-Adobe-2.0 EPSF-1.2\n\
%%%%BoundingBox: %d %d %d %d\n";

static char *multipage_header = "\
%%!PS-Adobe-2.0\n\
%%%%Pages: %d\n";

static char *ps_procs = "\
%%%%Orientation: Landscape\n\
%%%%Title: %s\n\
%%%%DocumentFonts: Times-Roman\n\
%%%%EndComments\n\
/S {stroke} bind def /L {lineto} bind def /M {moveto} bind def\n\
/rshow {dup stringwidth pop neg 0 rmoveto show} bind def\n\
/X {moveto XH2 0 rmoveto XH neg 0 rlineto XH2 0 rmoveto\n\
    0 XH2 rmoveto 0 XH neg rlineto 0 XH2 rmoveto} bind def\n\
/sf {setfont} bind def /drawbbox {4 dict begin\n\
   /y2 exch def /x2 exch def /y1 exch def /x1 exch def\n\
   x1 y1 X x1 y2 X x2 y2 X x2 y1 X\n\
end} bind def\n\
/drawrect {4 dict begin newpath\n\
   /y2 exch def /x2 exch def /y1 exch def /x1 exch def\n\
   x1 y1 moveto x1 y2 lineto x2 y2 lineto x2 y1 lineto closepath\n\
   stroke\nend} bind def\n\
/cr {currentpoint exch pop sub neg moveto} bind def\n\
/LS {\n\
   %d 0 translate 90 rotate newpath} bind def\n\
/C {curveto} bind def\n\
/XH %d def /XH2 %d def\n";

/*
 * This sets up the multipage plots. The sheets are all of height pht and
 * width pwt (in portrait mode). To allow for the fudge factor we do a
 * translate first.
 */
static char *multipage_procs = "\
/plarge {\n\
  /label exch def /i exch def /j exch def /bproc exch def\n\
  fy neg fx translate\n\
  F sf m dup pht add moveto label show\n\
  m m moveto 0 pht rlineto pwt 0 rlineto 0 pht neg rlineto closepath\n\
  gsave 0 setlinewidth stroke grestore clip\n\
  pwt j mul neg pht i mul neg translate bproc showpage\n\
} def\n\
/top m pht add def\n\
/right m pwt add def\n";

void init_output(char *s)
{
   float tx, ty, vsizeup, vsizedn, titles;

   titles = TITLE_SIZE + VS * (1 + nspars + (iflag != 0) + (tflag != 0));

   if(tflag)
      vsizeup = vsizedn = templ_h.pts;
   else {
      vsizeup = c_height + CROSS_SIZE/2;
      vsizedn = -c_depth + CROSS_SIZE/2;
   }

   /*
    * Here, tx is just a hack, an arbitrary value. ty is chosen so that the
    * plot is centered on the virtual canvas: if the canvas height is H,
    * then the value of ty is:
    *    (H - v_up - v_dn - titles)/2 + v_dn + titles
    */
   tx = templ_le.pts + TITLE_SIZE;
   ty = (nsheets_v*PAGEHEIGHT - vsizeup + vsizedn + titles)/2;
   
   if(multipage)
      fprintf(fout, multipage_header, nsheets_v * nsheets_h);
   else
      fprintf(fout, ps_header,
	      PS_WIDTH - (int)(ty + vsizeup + yfudge.pts),
	      (int)(tx - CROSS_SIZE/2 + xfudge.pts),
	      PS_WIDTH - (int)(ty - vsizedn - titles -CROSS_SIZE/2+yfudge.pts),
	      (int)(tx + xfudge.pts + chord.pts + CROSS_SIZE/2));
      
   fprintf(fout, ps_procs, s, PS_WIDTH, CROSS_SIZE, CROSS_SIZE/2);

   if(multipage) {
      fprintf(fout,
	      "/m %d def /pht %d def /pwt %d def /fx %.1f def /fy %.1f def\n",
	      MARGIN, PAGEWIDTH, PAGEHEIGHT, xfudge.pts, yfudge.pts);
      fputs(multipage_procs, fout);
   }
   else
      fprintf(fout, "%.1f %.1f translate\n", xfudge.pts, yfudge.pts);
   
   put_title(s);

   fprintf(fout, "/BoP { LS %.4g %.4g translate 0.1 setlinewidth} bind def\n",
	   tx, ty);
}

void put_curve(x1, y1, x2, y2, x3, y3)
double x1, y1, x2, y2, x3, y3;
{
   x1 *= chord.pts; y1 *= chord.pts; x2 *= chord.pts; y2 *= chord.pts;
   x3 *= chord.pts; y3 *= chord.pts;
   fprintf(fout, "%.2f %.2f %.2f %.2f %.2f %.2f C\n", x1, y1, x2, y2, x3, y3);
}

void init_lines(x, y)
double x, y;
{
   x *= chord.pts;
   y *= chord.pts;

   fprintf(fout, "/P {\nnewpath\nf sf 20 3 M (%s) show\n", info);
   fprintf(fout, "%.2f %.2f M\n", x, y);

}

void put_line(x, y)
double x, y;
{
   fprintf(fout, "%.2f %.2f L\n", x*chord.pts, y*chord.pts);
}

void draw_double_line(float x1, float y1, float x2, float y2, float spacing)
{
   fprintf(fout, "gsave newpath %.2f %.2f M %.2f %.2f L\n",
	   x1*chord.pts, y1*chord.pts, x2*chord.pts, y2*chord.pts);
   if(spacing >= 1.0) {
      fprintf(fout,
	      "gsave %.1f setlinewidth 0 setgray stroke\n", spacing);
      fprintf(fout,
	      "grestore %.1f setlinewidth 1 setgray stroke grestore\n",
	      spacing-0.5);
   }
   else
      fputs("stroke\n", fout);
}

void draw_template(float h, float le, float te)
{
   fputs("/Tp {\n", fout);
   fprintf(fout, "-%.2f -%.2f %.2f %.2f drawrect\n", le, h, te+chord.pts, h);
   fprintf(fout, "%.2f -%.2f M (%s) rshow\n", te+chord.pts-2, h-4.0, info);
   fprintf(fout, "%.2f %.2f M (%s) rshow\n", te+chord.pts-2, h-8.0, info);
   fputs("} bind def\n", fout);
}

static void put_title(char *s)
{
   char *p;
   int i, nlines=0;
   float offset;
   
   p = s;
   while(*p != '\n' && *p != '\0') p++;
   *p = '\0';

   fprintf(fout, "/vs %.4f def\n/SC{show lm vs cr} def\n", VS);
   fprintf(fout, "/Times-Roman findfont dup dup %d scalefont /B exch def\n",
	   TITLE_SIZE);
   fprintf(fout, "%d scalefont /F exch def 8 scalefont /f exch def\n", PS);

   if(aflag) {
      offset = chord.pts/3.0;
      fputs(" /savedmatrix matrix def /t {savedmatrix currentmatrix\n", fout);
      fprintf(fout, "%.3g 0.0 translate ", offset);
      fprintf(fout, "%.2g rotate ", angle);
      fprintf(fout, "%.3g 0.0 translate} bind def ", -offset);
      fputs("/et { savedmatrix setmatrix } bind def\n", fout);
   }
   else
      fputs("/et{}def/t{}def\n", fout);

   fputs("/T {\n   currentpoint pop /lm exch def\n", fout);
   fprintf(fout, "   0 %d rmoveto\n   B sf (%s) SC\n", -TITLE_SIZE, s);
   fprintf(fout, "   F sf\n   (Chord: %.3g%s) SC\n",
	   chord.raw_value, chord.units);
   nlines++;
   if(iflag) {
      fprintf(fout,
	     "   (with %.3g%s sheeting allowance) SC\n",
	     sheeting.raw_value, sheeting.units);
      nlines++;
      sprintf(info, "%s / %.3g%s / %.3g%s", s, chord.raw_value, chord.units,
	      sheeting.raw_value, sheeting.units);
   }
   else
      sprintf(info, "%s / %.3g%s", s, chord.raw_value, chord.units);

   if(tflag) {
      fprintf(fout, "   (template: height %.3g%s",
	     templ_h.raw_value, templ_h.units);
      fprintf(fout, ", %.3g%s at LE,", templ_le.raw_value, templ_le.units);
      fprintf(fout, " %.3g%s at TE) SC\n", templ_te.raw_value,
	      templ_te.units);
      nlines++;
   }

   if(aflag) {
      fprintf(fout, "   (Twist: %.2g\\312)SC\n", angle);
      nlines++;
   }

   for(i = 0; i < nspars; i++, nlines++)
      fprintf(fout,
	      "   (Spar at %.3g %% chord, thickness %.3g%s)SC\n",
	      spars[i].offset*100, spars[i].thick.raw_value,
	      spars[i].thick.units);
   
   fputs("} bind def\n", fout);

}

void end_output(point_t *points, int npoints)
{
   int i, j;

   fputs("} bind def\n\n", fout);

   if(multipage)
      fputs("/fproc {\n", fout);
   else
      fputs("%%EndProlog\n\n", fout);
   
   fprintf(fout, "BoP 0 %d neg M T t P et pathbbox newpath drawbbox stroke\n",
	   -(int)c_depth);
   if(tflag)
      fputs("Tp\n", fout);
   fprintf(fout,
	   "gsave[3 3]0 setdash %.4g 0 moveto %.4g 0 lineto stroke grestore\n",
	   -templ_le.pts, chord.pts+templ_te.pts);

   fputs("gsave t P clip\n", fout);
   for(i = 0; i < nspars; i++)
      draw_spar(spars[i].offset, spars[i].thick.pts, points, npoints);
   if(aflag)
      fprintf(fout,
	     "[3 3]0 setdash 0 setlinewidth 0 0 moveto %.2g 0 lineto stroke\n",
	      chord.pts);
   fputs("grestore\n", fout);

   if(iflag) {
      fprintf(fout, "gsave t P clip 0 setgray %.4g setlinewidth P stroke\n",
	      sheeting.pts*2);
      fprintf(fout, "%.4g setlinewidth 1 setgray P stroke\n",
	      sheeting.pts*2-0.5);
      fputs("grestore gsave [2 4] 0 setdash t P stroke grestore\n", fout);
   }
   else
      fputs("t P et stroke\n", fout);
   
   if(multipage) {
      int n=1;
      fputs("} bind def\n%%EndProlog\n\n", fout);
      for(i = 0; i < nsheets_v; i++)
	 for(j = 0; j < nsheets_h; j++,n++)
	    fprintf(fout, "%%%%Page: %d %d\n{fproc} %d %d((%d,%d)) plarge\n\n",
		    n, n, i, j, i+1, j+1);
      fputs("%%EOF\n\n", fout);
   }
   else
      fputs("showpage\n%%EOF\n", fout);
   fflush(fout);
}


static void draw_spar(float offset, float thick, point_t *points, int n)
{
   float upper=0.0, lower=0.0;
   int i;

   i = 0;
   while(i < n) {
      if(offset >= points[i].x) {
	 upper = points[i].y +
	    (points[i].y - points[i-1].y)/(points[i].x - points[i-1].x)*
	       (points[i].x - offset);
	 break;
      }
      i++;
   }
   while(i < n) {
      if(offset <= points[i].x) {
	 lower = points[i].y +
	    (points[i].y - points[i-1].y)/(points[i].x - points[i-1].x)*
	       (points[i].x - offset);
	 break;
      }
      i++;
   }

   draw_double_line(offset, 1.5*upper, offset, 1.5*lower, thick);
   
}
