/*
<applet code="DynaDraw" width=480 height=480>
</applet>
*/

import java.awt.*;
import java.applet.Applet;

/*
 *   dynadraw -
 *
 *        Use a simple dynamics model to create calligraphic strokes.
 *
 *   leftmouse   - used for drawing
 *   Home key    - clears page
 *   End key     - toggles fixed pen angle
 *
 *   uparrow     - wider strokes
 *   downarrow   - narrower strokes
 *
 *   shift down  - draw with white instead of black
 *
 *                  Paul Haeberli - 1989
 *
 *                Patrick Naughton - 1996
 *                    (Java Version)
 */
class filter {
    float curx, cury;
    float velx, vely, vel;
    float accx, accy;
    float angx, angy;
    float lastx, lasty;
    boolean fixedangle;
    final float EPSILON = 0.000001f;

    void setpos(float x, float y) {
        curx = x;
        cury = y;
        lastx = x;
        lasty = y;
        velx = 0.0f;
        vely = 0.0f;
        accx = 0.0f;
        accy = 0.0f;
    }

    float flerp(float f0, float f1, float p) {
        return ((f0*(1.0f-p))+(f1*p));
    }

    boolean apply(float mx, float my, float curmass, float curdrag) {

        /* calculate mass and drag */
        float mass = flerp(3.0f,30.0f,curmass);
        float drag = flerp(0.00f,0.5f,curdrag*curdrag);

        /* calculate force and acceleration */
        float fx = mx - curx;
        float fy = my - cury;

        float acc = (float)Math.sqrt(fx*fx+fy*fy);
        if(acc<EPSILON)
            return false;
        accx = fx/mass;
        accy = fy/mass;

        /* calculate new velocity */
        velx += accx;
        vely += accy;
        vel = (float)Math.sqrt(velx*velx+vely*vely);
        if(vel<EPSILON)
            return false;

        /* calculate angle of drawing tool */
        if (fixedangle) {
            angx = -0.5f;
            angy =  0.5f;
        } else {
            angx = -vely / vel;
            angy =  velx / vel;
        }

        /* apply drag */
        velx *= (1.0f-drag);
        vely *= (1.0f-drag);

        /* update position */
        lastx = curx;
        lasty = cury;
        curx += velx;
        cury += vely;

        return true;
    }
}


public class DynaDraw extends Applet implements Runnable {
    Graphics canvasG, myG;
    Image canvasimage;
    Font font;
    FontMetrics fm;
    Color pencolor;

    static final int SLIDERHIGH = 15;
    static final int SLIDERLEFT = 200;
    static final int DRAW = 0;
    static final int MASS_ADJUST = 1;
    static final int DRAG_ADJUST = 2;
    int mode;

    int xsize, ysize;
    float width;
    float odelx, odely;
    float curmass, curdrag;
    int xPoints[] = new int[4];
    int yPoints[] = new int[4];
    filter mouse = new filter();
    boolean mousePressed = false;
    float mx, my;

    public DynaDraw() {
    }

    float paramval(int x) {
	float p = (float)(x-SLIDERLEFT)/(xsize-SLIDERLEFT);
	if(p<0.0f)
	    return 0.0f;
	if(p>1.0f)
	    return 1.0f;
	return p;
    }

    void clearscreen() {
	canvasG.setColor(Color.white);
	canvasG.fillRect(0,0, xsize, ysize);
	showsettings();
    }

    void showsettings() {
	int xpos;
	int fd = fm.getDescent();
	canvasG.setColor(Color.gray);
	canvasG.fillRect(0,0,xsize, 2*SLIDERHIGH);
	canvasG.setColor(Color.black);
	canvasG.drawString("Drag " + curdrag, 20, SLIDERHIGH-fd);
	canvasG.drawString("Mass " + curmass, 20, 2*SLIDERHIGH-fd);
	canvasG.drawLine(SLIDERLEFT, 0, SLIDERLEFT, 2*SLIDERHIGH);
	canvasG.drawLine(0, SLIDERHIGH, xsize, SLIDERHIGH);
	canvasG.drawLine(0, 2*SLIDERHIGH, xsize, 2*SLIDERHIGH);
	canvasG.setColor(Color.red);
	xpos = (int) (SLIDERLEFT + curdrag * (xsize - SLIDERLEFT));
	canvasG.fillRect(xpos, 0, 4, SLIDERHIGH);
	xpos = (int) (SLIDERLEFT + curmass * (xsize - SLIDERLEFT));
	canvasG.fillRect(xpos, SLIDERHIGH, 4, SLIDERHIGH);
	canvasG.setColor(Color.black);
    }

    private int sy(float y) {
	return (int)(y*ysize);
    }

    void drawsegment(filter f) {
	float delx, dely;
	float wid;
	float px, py, nx, ny;

	wid = 0.04f - 0.3f*f.vel;
	wid *= width;
	if(wid < 0.00001f)
	    wid = 0.00001f;
	delx = f.angx * wid;
	dely = f.angy * wid;

	px = f.lastx;
	py = f.lasty;
	nx = f.curx;
	ny = f.cury;

	xPoints[0] = (int)(xsize*(px+odelx));
	xPoints[1] = (int)(xsize*(px-odelx));
	xPoints[2] = (int)(xsize*(nx- delx));
	xPoints[3] = (int)(xsize*(nx+ delx));

	yPoints[0] = (int)(ysize*(py+odely));
	yPoints[1] = (int)(ysize*(py-odely));
	yPoints[2] = (int)(ysize*(ny- dely));
	yPoints[3] = (int)(ysize*(ny+ dely));

	canvasG.setColor(pencolor);

	// draw line too so thin polys don't disappear.
	canvasG.drawLine((int)(xsize*nx), (int)(ysize*ny),
			 (int)(xsize*px), (int)(ysize*py));

	canvasG.fillPolygon(xPoints, yPoints, 4);

	updateimage(myG);

	odelx = delx;
	odely = dely;
    }

    public void updateimage(Graphics g) {    
	g.drawImage(canvasimage, 0, 0, this);
    }

    public void paint(Graphics g) {
	update(g);
    }

    public void update(Graphics g) {
	showsettings();
	updateimage(g);
    }

    public boolean mouseDown(Event evt, int x, int y) {
	if(y>0*SLIDERHIGH && y<2*SLIDERHIGH) {
	    if(y>SLIDERHIGH) {
		mode = MASS_ADJUST;
	    } else {
		mode = DRAG_ADJUST;
	    }
	} else {
	    mode = DRAW;
	    mousePressed = true;
	    pencolor = evt.shiftDown() ? Color.white : Color.black;
	    mx = x/(float)xsize;
	    my = y/(float)ysize;
	    startDrawing();
	}
	mouseDrag(evt, x, y);
	return true;
    }

    public boolean mouseDrag(Event evt, int x, int y) {
	float p;
	switch (mode) {
	  case MASS_ADJUST:
	    p = paramval(x);
	    if(p != curmass) {
		curmass = p;
		repaint();
	    }
	    break;
	  case DRAG_ADJUST:
	    p = paramval(x);
	    if(p != curdrag) {
		curdrag = p;
		repaint();
	    }
	    break;
	  case DRAW:
	    mx = x/(float)xsize;
	    my = y/(float)ysize;
	    break;
	}
	return true;
    }

    private synchronized void waitForPress() {
	try wait(); catch (InterruptedException e);
    }

    private synchronized void startDrawing() {
	notify();
    }

    private void sleep(int n) {
	try Thread.sleep(n); catch (Exception e);
    }

    public boolean mouseUp(Event evt, int x, int y) {
	mousePressed = false;
	return true;
    }

    public void run() {
	while(true) {
	    if (mousePressed) {
		mouse.setpos(mx,my);
		odelx = 0.0f;
		odely = 0.0f;
	    }
	    while (mousePressed) {
		mouse.apply(mx, my, curmass, curdrag);
		drawsegment(mouse);
		sleep(10); 
	    }
	    waitForPress();
	}
    }

    public void init() {
	curmass = 0.40f;
	curdrag = 0.46f;
	width = 1.5f;
	mouse.fixedangle = true;

	xsize = size().width;
	ysize = size().height;

	resize(xsize,ysize);

	canvasimage = createImage(xsize,ysize);
	myG = getGraphics();
	canvasG = canvasimage.getGraphics();

	font = new java.awt.Font("TimesRoman", Font.ITALIC, 14);
        canvasG.setFont(font);
        fm = canvasG.getFontMetrics();

	new Thread(this).start();
    }

    public boolean keyDown(Event evt, int key) {
	switch(key) {
	  case Event.UP:
	  case '.':
	  case '>':
	    width *= 1.414213;
	    break;
	  case Event.DOWN:
	  case ',':
	  case '<':
	    width /= 1.414213;
	    break;
	  case Event.END:
	  case ' ':
	    mouse.fixedangle = !mouse.fixedangle;
	    break;
	  case Event.HOME:
	  case '\r':
	  case '\n':
	    clearscreen();
	    break;
	}
	repaint();
	return true;
    }
}
